<?php

namespace FASTRANSPLUGIN\Element;

use Elementor\Controls_Manager;
use Elementor\Controls_Stack;
use Elementor\Group_Control_Typography;
use Elementor\Scheme_Typography;
use Elementor\Scheme_Color;
use Elementor\Group_Control_Border;
use Elementor\Repeater;
use Elementor\Widget_Base;
use Elementor\Utils;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Plugin;

/**
 * Elementor button widget.
 * Elementor widget that displays a button with the ability to control every
 * aspect of the button design.
 *
 * @since 1.0.0
 */
class Updated_Services extends Widget_Base {

	/**
	 * Get widget name.
	 * Retrieve button widget name.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'fastrans_updated_services';
	}

	/**
	 * Get widget title.
	 * Retrieve button widget title.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Our Services V3', 'fastrans' );
	}

	/**
	 * Get widget icon.
	 * Retrieve button widget icon.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string Widget icon.
	 */
	public function get_icon() {
		return 'eicon-library-open';
	}

	/**
	 * Get widget categories.
	 * Retrieve the list of categories the button widget belongs to.
	 * Used to determine where to display the widget in the editor.
	 *
	 * @since  2.0.0
	 * @access public
	 * @return array Widget categories.
	 */
	public function get_categories() {
		return [ 'fastrans' ];
	}
	
	/**
	 * Register button widget controls.
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function register_controls() {
		$this->start_controls_section(
			'updated_services',
			[
				'label' => esc_html__( 'Our Services V3', 'fastrans' ),
			]
		);
		$this->add_control(
			'subtitle',
			[
				'label'       => __( 'Sub Title', 'fastrans' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'dynamic'     => [
					'active' => true,
				],
				'placeholder' => __( 'Enter your Sub title', 'fastrans' ),
			]
		);
		$this->add_control(
			'title',
			[
				'label'       => __( 'Title', 'fastrans' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'dynamic'     => [
					'active' => true,
				],
				'placeholder' => __( 'Enter your title', 'fastrans' ),
			]
		);
		$this->add_control(
			'text_limit',
			[
				'label'   => esc_html__( 'Text Limit', 'fastrans' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 3,
				'min'     => 1,
				'max'     => 100,
				'step'    => 1,
			]
		);
		$this->add_control(
			'query_number',
			[
				'label'   => esc_html__( 'Number of post', 'fastrans' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 3,
				'min'     => 1,
				'max'     => 100,
				'step'    => 1,
			]
		);
		$this->add_control(
			'query_orderby',
			[
				'label'   => esc_html__( 'Order By', 'fastrans' ),
				'label_block' => true,
				'type'    => Controls_Manager::SELECT,
				'default' => 'date',
				'options' => array(
					'date'       => esc_html__( 'Date', 'fastrans' ),
					'title'      => esc_html__( 'Title', 'fastrans' ),
					'menu_order' => esc_html__( 'Menu Order', 'fastrans' ),
					'rand'       => esc_html__( 'Random', 'fastrans' ),
				),
			]
		);
		$this->add_control(
			'query_order',
			[
				'label'   => esc_html__( 'Order', 'fastrans' ),
				'label_block' => true,
				'type'    => Controls_Manager::SELECT,
				'default' => 'DESC',
				'options' => array(
					'DESC' => esc_html__( 'DESC', 'fastrans' ),
					'ASC'  => esc_html__( 'ASC', 'fastrans' ),
				),
			]
		);
		$this->add_control(
            'query_category', 
			[
			  'type' => Controls_Manager::SELECT,
			  'label' => esc_html__('Category', 'fastrans'),
			  'label_block' => true,
			  'options' => get_service_categories()
			]
		);
		$this->add_control(
            'show_bottom_info',
			[
				'label' => __( 'Enable/Disable Bottom Info Area', 'fastrans' ),
				'type'     => Controls_Manager::SWITCHER,
				'dynamic'     => [
					'active' => true,
				],
				'placeholder' => __( 'Enable/Disable Bottom Info Area', 'fastrans' ),
			]
		);
		$this->add_control(
			'text',
			[
				'label'       => __( 'text', 'fastrans' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXTAREA,
				'dynamic'     => [
					'active' => true,
				],
				'placeholder' => __( 'Enter your Section Bottom Description', 'fastrans' ),
			]
		);
		$this->add_control(
			'btn_title',
			[
				'label'       => __( 'Button Title', 'fastrans' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'dynamic'     => [
					'active' => true,
				],
				'placeholder' => __( 'Enter your Button Title', 'fastrans' ),
			]
		);
		$this->add_control(
            'btn_link',
			[
				'label' => __( 'Button Url', 'fastrans' ),
				'type' => Controls_Manager::URL,
				'label_block' => true, 
				'placeholder' => __( 'https://your-link.com', 'plugin-domain' ),
				'show_external' => true,
				'default' => [
					'url' => '',
					'is_external' => true,
					'nofollow' => true,
				],
			]
        );
		$this->end_controls_section();
	}

	/**
	 * Render button widget output on the frontend.
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since  1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
        $allowed_tags = wp_kses_allowed_html('post');
		
        $paged = get_query_var('paged');
		$paged = fastrans_set($_REQUEST, 'paged') ? esc_attr($_REQUEST['paged']) : $paged;

		$this->add_render_attribute( 'wrapper', 'class', 'templatepath-fastrans' );
		$args = array(
			'post_type'      => 'service',
			'posts_per_page' => fastrans_set( $settings, 'query_number' ),
			'orderby'        => fastrans_set( $settings, 'query_orderby' ),
			'order'          => fastrans_set( $settings, 'query_order' ),
			'paged'         => $paged
		);
		
		if( fastrans_set( $settings, 'query_category' ) ) $args['service_cat'] = fastrans_set( $settings, 'query_category' );
		$query = new \WP_Query( $args );

		if ( $query->have_posts() ) 
		{ ?>
		
        <!-- Updated Services Section -->
        <section class="updated-services-section">
            <div class="container">
                <?php if( $settings['subtitle'] || $settings['title'] ){ ?>
                <div class="ft-section-title-3 headline text-center">
                    <?php if( $settings['subtitle'] ){ ?><span class="text-uppercase"><?php echo wp_kses($settings['subtitle'], true)?></span><?php } ?>
                    <?php if( $settings['title'] ){ ?><h2><?php echo wp_kses($settings['title'], true)?></h2><?php } ?>
                </div>
                <?php } ?>
                
                <!-- Blocks Outer -->
                <div class="blocks-outer">
                    <div class="row clearfix">
                        
                        <!-- Updated Service Block -->
                        <?php while ( $query->have_posts() ) : $query->the_post(); ?>
                        <div class="updated-service-block col-lg-3 col-md-6 col-sm-12">
                            <div class="inner-box">
                                <div class="color-layer"></div>
                                <i class="icon <?php echo wp_kses(str_replace( "icon ",  "",  get_post_meta(get_the_id(), 'service_icon', true )), $allowed_tags); ?>"></i>
                                <i class="big-icon <?php echo wp_kses(str_replace( "icon ",  "",  get_post_meta(get_the_id(), 'service_icon', true )), $allowed_tags); ?>"></i>
                                <h3><a href="<?php echo esc_url(get_post_meta( get_the_id(), 'service_url', true ));?>"><?php the_title(); ?></a></h3>
                                <div class="text"><?php echo wp_kses(fastrans_trim(get_the_content(), $settings['text_limit']), true); ?></div>
                            </div>
                            <div class="more-btn position-absolute">
                                <a class="" href="<?php echo esc_url(get_post_meta( get_the_id(), 'service_url', true ));?>"><?php esc_html_e('Read more', 'fastrans'); ?> <i class="far fa-arrow-right"></i></a>
                            </div>
                        </div>
                        <?php endwhile; ?>
                    </div>
                    
                    <!-- Lower Box -->
                    <?php if($settings['show_bottom_info']){ ?>
                    <div class="lower-box">
                        <?php if($settings['text'] || $settings['btn_link']['url'] || $settings['btn_title']){ ?>
                        <div class="text">
                        <?php echo wp_kses($settings['text'], true)?> 
                        <a href="<?php echo esc_url($settings['btn_link']['url'])?>"><?php echo wp_kses($settings['btn_title'], true)?></a>
                        </div>
                        <?php } ?>
                    </div>
                    <?php } ?>
                    
                </div>
                <!-- End Blocks Outer -->
                
            </div>
        </section>
        <!-- End Updated Services Section -->
                    
        <?php }
		wp_reset_postdata();
	}
}